///////////////////////////////////////////////////////////////
//
// RecommendationManager.java
//
//		Basic implementation of the IRecommendationManager
//		interface.
//
package Alkindi.Services.ServicesImpl;

import java.util.*;
import java.sql.*;
import java.math.*;
import javax.ejb.*;
import javax.transaction.*;
import oracle.jdbc.driver.*;
import Alkindi.Data.*;
import Alkindi.Services.*;
import Alkindi.Services.Util.*;
import Alkindi.Services.InternalData.*;
import Alkindi.Services.BeanIface.IAccountManagerBean;

/* 
$Header: RecommendationManager.java, 51, 6/7/01 4:33:49 PM, Schwartz, Joe$
$Log: 
 51   Alkindi Development1.50        6/7/01 4:33:49 PM    Schwartz, Joe  
      Reinstated old alkindex calculation.
 50   Alkindi Development1.49        5/10/01 12:02:44 PM  Schwartz, Joe   Made
      getAlkindex to spec. Removed dependence of getRecs on MyList data.
 49   Alkindi Development1.48        5/8/01 9:56:14 AM    Schwartz, Joe  
      Changed use of InternalEJBManager to InternalComponentManager. Removed
      dependencies on EJB from these classes. Cleaned up comments.
 48   Alkindi Development1.47        5/3/01 10:11:21 AM   Schwartz, Joe  
      Removed some variables useful only for stateful operation. Removed use of
      class RecWithScoringFunction; changed call to insertRecs to use ordinary
      recList . 
 47   Alkindi Development1.46        4/26/01 3:25:10 PM   Schwartz, Joe  
      Chagned to use the new attribute of ProdRecData that provides the overall
      average rating for a product when creating Rec objects to return in
      getRecs.
 46   Alkindi Development1.45        4/26/01 2:37:17 PM   Schwartz, Joe  
      Modifed to account for new InternalData package.
 45   Alkindi Development1.44        4/26/01 1:11:15 PM   Schwartz, Joe   Moved
      into new ServicesImpl package.
 44   Alkindi Development1.43        3/23/01 4:58:05 PM   Schwartz, Joe   Made
      change to support stateless RecMgr.
 43   Alkindi Development1.42        3/14/01 5:50:43 PM   Schwartz, Joe   Made
      the rest of these implementation classes abstract.
 42   Alkindi Development1.41        3/14/01 5:49:03 PM   Schwartz, Joe  
      Removed sessCtx member.
 41   Alkindi Development1.40        3/2/01 4:59:28 PM    Schwartz, Joe  
      Modified for separate database search for Primary products.
 40   Alkindi Development1.39        2/27/01 1:01:56 PM   Schwartz, Joe  
      Working on optimizing stateful bean.
 39   Alkindi Development1.38        2/23/01 7:25:16 PM   Schwartz, Joe  
      Working on cache issues.
 38   Alkindi Development1.37        2/23/01 4:32:43 PM   Schwartz, Joe  
      Changed getPredictedRating to use new SP and return a float. Added state
      caching of PCWeights when all PCs are requested. Numerous small fixes. 
 37   Alkindi Development1.36        2/22/01 3:18:06 PM   Schwartz, Joe  
      Several changes to support new spec. Made stateful bean.
 36   Alkindi Development1.35        2/19/01 6:16:32 PM   Schwartz, Joe  
      Interin checkin: working on new RecMgr spec.
 35   Alkindi Development1.34        2/16/01 5:51:44 PM   Schwartz, Joe  
      Interim save: working off of new RecMgr spec. Adapting to use new objects
      from database. 
 34   Alkindi Development1.33        2/12/01 4:47:45 PM   Schwartz, Joe  
      Changed to account for Product ID becoming an int.
 33   Alkindi Development1.32        12/29/00 1:14:11 PM  Schwartz, Joe  
      Removed dependency on RatingMgrDBUtils. Replaced with dependency on
      UserStat.
 32   Alkindi Development1.31        12/28/00 2:54:03 PM  Schwartz, Joe  
      Standardized methods to use final parameters and fully-qualified types
      where necessary. Makes Rose happy.
 31   Alkindi Development1.30        12/28/00 1:46:06 PM  Schwartz, Joe  
      Replaced getRecommendations() body, which went missing due to a Rose
      error.
 30   Alkindi Development1.29        12/28/00 1:10:58 PM  Schwartz, Joe   Added
      Version Control header info.
      Synchronized with IRecommendationManager interface.
      Changed package-scope members to protected.
 29   Alkindi Development1.28        12/26/00 6:57:16 PM  Schwartz, Joe   
 28   Alkindi Development1.27        12/26/00 5:37:00 PM  Schwartz, Joe   
 27   Alkindi Development1.26        12/20/00 5:44:57 PM  Schwartz, Joe   
 26   Alkindi Development1.25        12/19/00 3:51:58 PM  Schwartz, Joe   
 25   Alkindi Development1.24        12/18/00 6:08:25 PM  Schwartz, Joe   
 24   Alkindi Development1.23        12/18/00 4:35:39 PM  Schwartz, Joe   
 23   Alkindi Development1.22        12/18/00 12:05:40 PM Schwartz, Joe   Moved
      internal data classes to Utils package & regenerated classes from Rose.
 22   Alkindi Development1.21        12/18/00 11:22:09 AM Schwartz, Joe   
 21   Alkindi Development1.20        12/7/00 10:55:39 AM  Schwartz, Joe  
      Reduced debug log notifications.
 20   Alkindi Development1.19        12/6/00 7:11:58 PM   Schwartz, Joe   Added
      catch blocks for generic exceptions and finally blocks to close all
      database connections.
 19   Alkindi Development1.18        12/6/00 11:27:59 AM  Schwartz, Joe   
 18   Alkindi Development1.17        12/4/00 2:36:45 PM   Schwartz, Joe   Really
      fixed the array out of bounds error in getRecommendations.
 17   Alkindi Development1.16        12/4/00 2:31:40 PM   Schwartz, Joe   Fixed
      array out of bounds in getRecommendations() for users with no ratings in
      dB.
 16   Alkindi Development1.15        12/3/00 11:09:37 PM  Schwartz, Joe   Made
      several changes to getRecommendations per DM's guidance. Added changes
      from several CRs as well. See code comments for details.
 15   Alkindi Development1.14        12/3/00 10:40:17 PM  Schwartz, Joe   
 14   Alkindi Development1.13        12/1/00 12:23:17 PM  Schwartz, Joe   
 13   Alkindi Development1.12        11/30/00 12:46:43 AM Schwartz, Joe   
 12   Alkindi Development1.11        11/29/00 1:32:16 PM  Schwartz, Joe   Fixed
      a bunch of small problems in how it recommends.
 11   Alkindi Development1.10        11/20/00 10:52:35 PM Schwartz, Joe   
 10   Alkindi Development1.9         11/14/00 12:53:15 PM Schwartz, Joe   
 9    Alkindi Development1.8         11/8/00 2:59:04 PM   Schwartz, Joe   
 8    Alkindi Development1.7         11/7/00 2:04:33 PM   Schwartz, Joe  
      Modified to close DB connections.
 7    Alkindi Development1.6         11/6/00 7:05:59 PM   Schwartz, Joe   
 6    Alkindi Development1.5         11/6/00 11:50:21 AM  Schwartz, Joe   
 5    Alkindi Development1.4         10/22/00 10:38:46 AM Schwartz, Joe  
      AppianDelivery 10.20.00
 4    Alkindi Development1.3         10/20/00 12:00:44 PM Schwartz, Joe   
 3    Alkindi Development1.2         10/17/00 2:22:57 PM  Schwartz, Joe  
      Delivery 10.16.00
 2    Alkindi Development1.1         10/7/00 4:34:33 PM   Schwartz, Joe   
 1    Alkindi Development1.0         10/7/00 4:26:03 PM   Schwartz, Joe   
$
$NoKeywords$
 */

/**
 */
public abstract class RecommendationManager implements IRecommendationManager 
{
	
	/**
	 * Caches the lists of all the ProdRecData objects retrieved. Used to minimize database calls priorizing operations. Uses the Integer objects as keys, and ArrayList objects as values.
	 */
	protected HashMap prdListMap = new HashMap();
	
	/**
	 * PC weight equation term.
	 */
	protected static double propA;
	
	/**
	 * PC weight equation term.
	 */
	protected static double propB;
	protected static boolean propDebugPCWeights = false;
	protected static int propEvalDontKnow;
	protected static int propEvalInterested;
	protected static int propEvalNotInterested;
	
	/**
	 * PC weight equation term.
	 */
	protected static double propg;
	
	/**
	 * PC weight equation term.
	 */
	protected static double proph;
	
	/**
	 * PC weight equation term.
	 */
	protected static double propK = 0;
	
	/**
	 * 	Number of Product Clusters (an Engine property).
	 */
	protected static int propNumPCs;
	
	/**
	 * Used for assigning priorites to candidate products for recommending.
	 */
	protected static double propPRlim0;
	
	/**
	 * Used for assigning priorites to candidate products for recommending.
	 */
	protected static double propPRlim1;
	
	/**
	 * Used for assigning priorites to candidate products for recommending.
	 */
	protected double propPRlim2;
	
	/**
	 * Maximum value for denominator in PC Weight equations.
	 */
	protected static double propPsi;
	protected static int propTint_dk_0;
	protected static int propTrate;
	
	/**
	 * Minimum number of days ago a Product may have been recommended  to a given User and not be excluded for further recommending.
	 */
	protected static int propTshown;
	protected double propUCmin_rate;
	
	/**
	 * Threshold rating used in selecting candidate Products for recommending.
	 */
	protected static double propUSERmin_rate;
	
	/**
	 * Class name used in logging.
	 */
	static final String cName = "RecommendationManager";
	
	/**
	 * Logging object
	 */
	LogManager logger = null;
	RecMgrDBUtils recMgrDB = null;
	
	/**
	 * Returns the users Alkindex as a float.
	 * @param user The SystemUser whose Alkindex is to be retrieved.
	 * @return A float representing the user's Alkindex.
	 * @throws AlkExcept
	 * @roseuid 3A4B661100AB
	 */
	public float getAlkindex(final Alkindi.Data.SystemUser user) throws AlkExcept 
	{
		final String mName = "getAlkindex";
		try {
			float Alkindex = 0;
			double propAlkindexC = PropertyManager.getDouble("alkindex.C");
			double numEval = 0;
			try {
				numEval = UserStat.getNumProdSeen(user);
			}
			catch (Exception e) {
				logger.err(cName, mName, "getNumProdSeen() for user id=" + user.id + " threw exception: " + e.toString());
			}
			double scale1 = PropertyManager.getDouble("alkindex.scaler1");
			double scale2 = PropertyManager.getDouble("alkindex.scaler2");
			double max = PropertyManager.getDouble("alkindex.max");
			Alkindex = (float)(max - scale1 * Math.exp(-1D * scale2 * numEval));
			return Alkindex;
		}
/*

			double w = 0;
			double h = 0;
			double nn = 0;
			double BB = 0;
			int ff = 0;
			double bb = 0;
			ArrayList uusdList = recMgrDB.getUserUCSubgroupStats(user);
			double numerator = 0;
			double denominator = 0;
			for (int idx = uusdList.size(); --idx > -1; ) {
				UserUCSubgroupData uusd = (UserUCSubgroupData)uusdList.get(idx);
				if (uusd.Bi < uusd.BBi) 
					uusd.Bi = uusd.BBi;
				if (uusd.fi < 0.02)
					uusd.Bi = uusd.BBi;
				numerator += uusd.ni * (uusd.BBi - uusd.Bi * Math.exp( -1 * propAlkindexC * uusd.fi));
				denominator += uusd.ni * uusd.BBi;
			}
			if (denominator > 0) 
				Alkindex = (float)(numerator/denominator);
//			if (Alkindex < 0) 
//				Alkindex = 0;
			return Alkindex;
		}*/
	/*	catch (SQLException se) {
			String errmsg = "SQLException in getAlkindex: " + se.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg, 6004);
		}*/
		catch (AlkExcept ae) {
			logger.err(cName, mName, ae.errCode() + ":" + ae.getMessage());
			throw ae;
		}
		catch(Exception e) {
			String errmsg = "Exception in getAlkindex: " + e.getMessage();
			logger.err(cName, mName, errmsg);
			throw new AlkExcept(errmsg, 2503);
		}
	}
	
	/**
	 * Returns the Product Cluster Weights for the given User. If all PCs are requested, this routine tris to use the cached PC weights.
	 * @param user The SystemUser for whom to get the PC weights.
	 * @param pcList The list of requested Product Clusters for the call to getRecommendations.
	 * @roseuid 3A83263E01A5
	 */
	protected double[] getPCWeights(final Alkindi.Data.SystemUser user, final Alkindi.Data.ProductClusterList pcList) throws AlkExcept 
	{
		final String mName = "getPCWeights";

		//	Declare return value for weights and zero it out.
		//	
		double[] pcWeights = new double[propNumPCs];
		Arrays.fill(pcWeights, 0d);
		
		//	Number of requested PCs
		//
		int numReqPC = pcList.size();
		
		//	Were all PCs requested?
		//
		boolean allPCsRequested = (numReqPC == propNumPCs); 
		
		if (propDebugPCWeights) 
			logger.dbgLog(cName, mName, "Calculating new PC Weights.");
			
		//	Calculate the PC weights.
		//	First get PC Stats for the user & overall evaluation stats.
		//
		UserEvalData userEvalData = null;
		HashMap pcDataMap = recMgrDB.getPCData(user, userEvalData);
		if (propDebugPCWeights)
			logger.dbgLog(cName, mName, "pcDataMap size: " + pcDataMap.size());

		//	First find out if the user has been clustered with respect to any of the
		//	requested PCs
		//
		boolean clusteredForAnyRequested = false;

		for (int idx = 0; idx < numReqPC; idx ++) {
			ProductCluster pc = pcList.get(idx);
			PCRecData data = (PCRecData)pcDataMap.get(pc);
			if (data.userClustered) {
				clusteredForAnyRequested = true;
				break;
			}
		}

		//	The following is a bit complicated but is easily viewed with the Rose
		//	SRS.
		//
		//	If user is NOT clustered with respect to any of the requested PCs, 
		//	then calc weights using equation 1 of the RecMgr FRS. 
		//	(Uses normalized # of Recommendable Products per requested PC).
		//
		//	Otherwise, 
		//		For each requested PC, if the user is clusterd with respect to the 
		//		PC, use Eqn 2 fro, the spec. 
		//		If the user is not clustered with respect to the PC, then use either 
		//		equation 3 or a zero weight, depending of whether all available PCs 
		//		were requested.
		//
		for (int pcIdx = 0; pcIdx < numReqPC; pcIdx++ ) {
			//	Retrieve PC and PCRecData objects
			//
			ProductCluster pc = pcList.get(pcIdx);
			PCRecData data = (PCRecData)pcDataMap.get(pc);
			//	User clusted with respect to some requested PC
			//
			if (clusteredForAnyRequested) {
				//	User clustered for the current PC: use eqn. 2, leaving out Alkindex term for now...
				//
				//	wi = ni*EXP(A*ri + B*epsiloni + K*Di + (g*Ii - h*Ni)/Mi)
				//
				if (data.userClustered) {
					float epsiloni = data.evalData.ratedToShown();
					double Mi = Math.max(data.evalData.numShown(), propPsi);
					pcWeights[pc.id - 1] = data.numRecProds * Math.exp(propA * data.avgEval + propB *epsiloni + (propg * data.evalData.numYes - proph * data.evalData.numNo) / Mi );	

				}
				else{ 
					//	User NOT clustered for current PC
					//	Check if all PCs were requested
					//
					if (allPCsRequested) {
						//	Use zero
						//
						pcWeights[pc.id - 1] = 0;
					}
					else {
						//	NOT All PCs requested: use eqn. 3, leaving out Alkindex term for now.
						//
						//	wi = ni*EXP(A*rall + B*epsilonall + K*Dall + (g*Iall - h*Nall)/Mall)
						//	
						//
						double Mall = Math.max(userEvalData.numShown(), propPsi);
						pcWeights[pc.id - 1] = data.numRecProds * Math.exp(propA * userEvalData.avgEval() + propB * userEvalData.ratedToShown() + (propg * userEvalData.numYes + proph * userEvalData.numNo) / Mall );
					}
				}	//	END if/else user clustered for current PC
			}
			else {		
				//	User NOT clustered for any requested PC
				//	Eqn. 1
				pcWeights[pc.id - 1] = data.numRecProds;
			}//			END if/else user clustered for any requested PC

		}//	END loop over PCs
		//	Free resources
		//
		userEvalData = null;
		pcDataMap.clear();
		pcDataMap = null;
	
		return pcWeights;
	}
	
	/**
	 * Retrieves the list of ProdRecData objects for the requested Product Cluster. Uses a cached list if possible, otherwise calls to the database and prioritizes a new product data list.
	 * @param pcID The Product Cluster ID for which to retrieve a list of Product data objects.
	 * @return An ArrayList of ProdRecDataObjects.
	 * @throws AlkExcept
	 * @roseuid 3A93ED4D0119
	 */
	protected java.util.ArrayList getPRDList(final Alkindi.Data.SystemUser usr, final int pcID, final boolean usePrimary) throws AlkExcept 
	{
		final String mName = "getPRDList";
		//	First, attempt to get the cached PRDList for the PC.
		//
		Integer key = new Integer(pcID);
		ArrayList cachedList = (ArrayList)prdListMap.get(key);
		if(cachedList != null) {
			logger.dbgLog(cName, mName, "Got list for PC=" + pcID + " from cache.");
			if (cachedList.size() > 0)
				return cachedList;
		}
		logger.dbgLog(cName, mName, "Getting list for PC=" + pcID + " from database.");

		//	If there was nothing in the cache, retrieve the list from the dB.
		//	The usePrimary parameter controls which kind of products are retrieved.
		//
		ArrayList prdList = null;
		if (usePrimary)
			prdList = recMgrDB.getPrimaryProductData(usr, pcID, 200);
		else
			prdList = recMgrDB.getProductData(usr, pcID, propUSERmin_rate, propTshown, 20);

		//	Prioritize the list
		//
	//	prioritizeProducts(prdList);
		//	Cache the list
		//
		prdListMap.put(key, prdList);
		return prdList;
	}
	
	/**
	 * This routine actually gets the predicted rating for a given product and user.
	 * @param user The User for whom to calculate the predicted rating.
	 * @param prod The product for which to calculate the predicted rating.
	 * @return The predicted rating as a float.
	 * @throws AlkExcept
	 * @roseuid 3A4B661100EA
	 */
	public float getPredictedRating(final Alkindi.Data.SystemUser user, final Alkindi.Data.Product prod) throws AlkExcept 
	{
		final String mName = "getPredictedRating";
		double predRating = 0D;

		try {
			return recMgrDB.getPredictedRating(user, prod);
		}
		catch(AlkExcept ae) {
			throw (AlkExcept)ae;
		}
		catch(Exception e) {
			throw new AlkExcept("Unexpected error in " + mName, 2513, e);
		}
	}
	
	/**
	 * @return RecommendationList
	 * @param user
	 * @param numrecommended
	 * @param productGroup
	 * @exception AlkExcept
	 * @roseuid 3A4B6611007E
	 */
	public final Alkindi.Data.RecommendationList getRecommendations(final Alkindi.Data.SystemUser user, final int numToRecommend, final Alkindi.Data.ProductClusterList pcList) throws AlkExcept 
	{
		final String mName = "getRecs";
		final int MAX_FAULTS=20;
		try {
			prdListMap.clear();
			//	If Instrumentation flag is set, initialize an instrumenter.
			//
			Instrumenter inst = null;
			boolean doInstrumentation = PropertyManager.getBool("Instrumentation");
			if (doInstrumentation) 
				inst = Instrumenter.getInstance(logger, cName, mName);
				
			RecommendationList recList = new RecommendationList();
			
			logger.log(cName, mName, "Recommending " + numToRecommend + " products for userid " + user.id);
			
			double[] pcWeights = getPCWeights(user, pcList);
			
			//	Make a copies PC weights for Priority1 and other search
			//
			if (doInstrumentation)
				inst.startTiming();
			double[] primaryPCWeights = new double[pcWeights.length];
			System.arraycopy(pcWeights, 0, primaryPCWeights, 0, pcWeights.length);
			if (doInstrumentation)
				inst.stopTiming();
			//	Debug: Dump weights
			//
			if (propDebugPCWeights) {
				StringBuffer wBuf = new StringBuffer("PC Weights for user " + user.id + "\n");
				wBuf.append(user.id);
				for (int idx = 0; idx < pcWeights.length; idx++) {
					wBuf.append("PC #");
					wBuf.append(idx);
					wBuf.append(": ");
					wBuf.append(pcWeights[idx]);
					wBuf.append("\n");
				}
				logger.dbgLog(cName, mName, wBuf.toString());
				wBuf = null;
			}
	
			//	Flag that indicates using Priority 1 products if possible.
			//
			boolean usingPrimary = true;

			double [] sfWts = recMgrDB.getScoreWeight();
						
			//	Begin loop to recommend a certain number of products
			//
			while ((recList.size() < numToRecommend)){
			
				//	Check if we should persist in trying to get Priority 1 products
				//
				if (usingPrimary) {
					usingPrimary = false;
					for (int idx = 0; idx < primaryPCWeights.length; idx ++) {
						if (primaryPCWeights[idx] > 0) {
							usingPrimary = true;
							break;
						}
					}
				}
				
				//	Set up convenient reference to current PC Weights array 
				//
				double[] activePCWeights = usingPrimary ? primaryPCWeights : pcWeights;
			
				//	Pick a cluster from which to recommend.
				//	RandomStick returns an int from 0-16; we add 1 to get a 
				//	correct PC number.
				//
				int pcID = RandomStick.chooseByWeights(activePCWeights) + 1;
				
				//	Pick a Scoring Function to use
				//
				int scoreFcnID = RandomStick.chooseByWeights(sfWts) + 1;
				logger.dbgLog(cName, mName, "Trying rec from PC #" + pcID + " SF #" + scoreFcnID);

				//	Get candidate Products for this PC
				//
				if (doInstrumentation)
					inst.startTiming();
					
				ArrayList prdList = getPRDList(user, pcID, usingPrimary);
				
				if (doInstrumentation)
					inst.stopTiming("Getting prod list for pc " + pcID);

				//	Remove products already selected for recommendations.
				//	
				ProdRecData prdUtil = new ProdRecData(0);
				for (int recIdx = recList.size(); --recIdx > -1; ) {
					prdUtil.productID = recList.get(recIdx).prod.id;
					int foundIdx = prdList.indexOf(prdUtil);
					if (foundIdx >= 0) 
						prdList.remove(foundIdx);
				}
				
				//	Check if we have run out of products from this PC
				//	If so, adjust appropriate weight & continue search.
				//
				if (prdList.size() == 0) {
					activePCWeights[pcID - 1] = 0;
					continue;
				}
				
				//	Prioritize and sort products. Only prioritize if using non-primary products.
				//
				if (doInstrumentation)
					inst.startTiming();
				
				if (!usingPrimary)
					prioritizeProducts(prdList, recList);
				
				if (doInstrumentation) {
					inst.stopTiming("Prioritize prods");
					inst.startTiming();
				}
				
				sortProducts(prdList, scoreFcnID);
				
				if (doInstrumentation) 
					inst.stopTiming("Sort prods");				

				logger.dbgLog(cName, mName, "After sort: " + prdList.size() + " products");
				
				//	For Primary Products, look for highest-scoring product with acceptable PR.
				//	For others, just take them in order of priority,
				//	
				ProdRecData selectedProdData = null;
				Product prodRec = new Product(0);
				if (usingPrimary) {
					boolean selectedProd = false;
					while (prdList.size() > 0 && !selectedProd) {					
						selectedProdData = (ProdRecData)prdList.remove(0);
						prodRec.id = selectedProdData.productID;
						if (getPredictedRating(user, prodRec) > propPRlim0) 
							selectedProd = true;
					}
					//	Zero out weight for empty PC
					//
					if (prdList.size() == 0) 
						primaryPCWeights[pcID-1] = 0;

					if (!selectedProd) {
						logger.dbgLog(cName, mName, "Primary Search: nothing for PC " + pcID);
						continue;
					}
					logger.dbgLog(cName, mName, "Primary Search: selected product " + selectedProdData.productID);
				}
				else {
					//	For rest of search, select products in sorted order, regardless of
					//	priority.
					//
					selectedProdData = (ProdRecData)prdList.remove(0);
					logger.dbgLog(cName, mName, "Secondary Search: selected product " + selectedProdData.productID);					
					//	Zero out weight for empty PC
					//
					if (prdList.size() == 0) 
						pcWeights[pcID-1] = 0;
				}
				prodRec.id = selectedProdData.productID;
				//Recommendation rec = new Recommendation(user, prodRec, selectedProdData.predRating, selectedProdData.avgProdEvalInUC);
				Recommendation rec = new Recommendation(user, prodRec, getPredictedRating(user, prodRec), selectedProdData.avgEval, scoreFcnID, pcID);
				
				//	Add to list to return.
				//
				recList.add(rec);
			} // while
			//logger.dbgLog(cName, mName, "Attempting to insert recs.");
			recMgrDB.insertRecommendations(recList);
			//logger.dbgLog(cName, mName, "Returning " + rl.size() + " items.");
			return recList;
		}
		catch (AlkExcept ae) {
			logger.err(cName, mName, ae.toString());
			throw ae;
		}
		catch (Exception e) {
			AlkExcept ae = new AlkExcept("Additional Exception", 2502, e);
			logger.err(cName, mName, ae.toString());
			throw ae;
		}
	}
	
	/**
	 * @roseuid 3A4B661100CB
	 */
	public Alkindi.Data.RecommendationList getRecommendations(final Alkindi.Data.SystemUser user, final int numDesired) throws AlkExcept 
	{
		//	Add all PCs to requested list, then call other getRecommendations()
		//
		ProductClusterList pcl = new ProductClusterList();
		for (int idx = 1; idx <= propNumPCs; idx ++) {
			ProductCluster pc = new ProductCluster(idx);
			pcl.add(pc);
		}
		return getRecommendations(user, numDesired, pcl);
	}
	
	/**
	 * 	@return void
	 * @roseuid 3A4B6611007D
	 */
	public void initLogging() 
	{
		if (logger == null)
			logger = new LogManager();
	}
	
	/**
	 * Assigns priorities to the Products as candidates for recommending..
	 * @param prodDataList An ArrayList of ProdRecData objects.
	 * @param recList The current list ofaccumulated recommendations.
	 * @return void
	 * @roseuid 3A917BCF037A
	 */
	protected void prioritizeProducts(java.util.ArrayList prodDataList, Alkindi.Data.RecommendationList recList) 
	{
		final long MILLIS_IN_DAY = 1000*3600*24;
		
		int pdListSize = prodDataList.size();
		
		for (int pdIdx = 0; pdIdx < pdListSize; pdIdx ++) {
			ProdRecData prData = (ProdRecData)prodDataList.get(pdIdx);
			prData.priority = 100;
			
			int daysSinceEval = -1;
			if (prData.	existingEvalTimestamp != null) {
				daysSinceEval = (int)Math.round( (Calendar.getInstance().getTime().getTime() - prData.existingEvalTimestamp.getTime()) / MILLIS_IN_DAY);
			}

			//	Priority 12: The PR for this product falls within the range PR < PRlim2 
			//	
			if (prData.predRating < propPRlim2) {
				prData.priority = 12;
				continue;
			}
			//	Priority 11: 11.The PR for this product falls within the range 
			//	PRlim2 <= PR < PRlim1 
			//
			if (prData.predRating >= propPRlim2 && prData.predRating < propPRlim1) {
				prData.priority = 11;
				continue;
			}
			
			//	Priorites 10 & 9
			//
			if (prData.existingEval >= propUSERmin_rate) {
				// 	Priority 10: The user has given this product at least a USERmin_rate 
				//	(tunable parameter) rating, and no more than Trate	days have elapsed since.
				//
				if (daysSinceEval <= propTrate) 
					prData.priority = 10;
				else 
					//	Priority 9: The user has given this product at least a USERmin_rate 
					//	(tunable parameter) rating, and more than Trate days have elapsed since.
					//
					prData.priority = 9;
				
				continue;
			}
			//	Priority 8 is skipped, since it depends on MyList, which is not an 
			//	application-independent feature.
			//
			//	Priority 7 skipped for now.
			//
			//
			//	Priority 6: When last shown the product, the user either gave no feedback 
			//	or selected "Interested" or "Don't Know"; and no more than Tint_dk_0 
			//	days have elapsed since.
			//
			if (daysSinceEval >= 0 
				&& (prData.existingEval == 0 || prData.existingEval == -4 || prData.existingEval == -2) 
				&& daysSinceEval <= propTint_dk_0) {
				
				prData.priority = 6;
				continue;
			}
			
			//	Priority 5: The PR for this product falls within the range 
			//	PRlim1 <= PR < PRlim0
			//
			if (prData.predRating >= propPRlim1 && prData.predRating < propPRlim0)  {
				prData.priority = 5;
			}
			//	Priority 4:	When last shown the product, the user either gave no feedback 
			//	or selected "Interested" or "Don't Know"; and more than Tint_dk_0 
			//	days have elapsed since.
			//
			if (daysSinceEval >= 0 
				&& (prData.existingEval == 0 || prData.existingEval == -4 || prData.existingEval == -2) 
				&& daysSinceEval > propTint_dk_0) {
				
				prData.priority =4;
				continue;
			}
			//	Priority 3: The UC has no ratings for the product.
			//
			if (prData.avgProdEvalInUC == 0) {
				prData.priority = 3;
				continue;
			}
			//	Priority 2: The average UC rating for this product is below UCmin_rate
			//
			if (prData.avgProdEvalInUC < propUCmin_rate) {
				prData.priority = 2;
				continue;
			}
			//	Priority 1: Everything else.
			//
			prData.priority = 1;

		}
	}
	
	/**
	 * 	@return void
	 * @roseuid 3A4B6611006D
	 */
	public void readProps() 
	{
		final String mName = "Constructor";
		try {
			propNumPCs = PropertyManager.getInt("NumPC");
			propA = PropertyManager.getDouble("A");
			propB = PropertyManager.getDouble("B");
			propK = PropertyManager.getDouble("K");
			propg = PropertyManager.getDouble("g");
			proph = PropertyManager.getDouble("h");
			propPsi = PropertyManager.getDouble("Psi");
						
			propPRlim0 = PropertyManager.getDouble("PRlim0");
			propPRlim1 = PropertyManager.getDouble("PRlim1");
			propPRlim2 = PropertyManager.getDouble("PRlim2");
			propTint_dk_0 = PropertyManager.getInt("Tint_dk_0");
			propTrate = PropertyManager.getInt("Trate");
			propTshown = PropertyManager.getInt("Tshown");
			propUCmin_rate = PropertyManager.getDouble("UCmin_rate");
			propUSERmin_rate = PropertyManager.getDouble("USERmin_rate");
			
			propEvalInterested = PropertyManager.getInt("EvalInterested");
			propEvalNotInterested = PropertyManager.getInt("EvalNotInterested");
			propEvalDontKnow = PropertyManager.getInt("EvalDontKnow");
			
			propDebugPCWeights = PropertyManager.getBool("debugPCWeights");
			
			recMgrDB = new RecMgrDBUtils();
		}
		catch(Exception e) {
			logger.err(cName, mName, e.getMessage());
			System.out.println("ERROR in " + cName + "." + mName + ": " + e.getMessage());
			//e.printStackTrace();
		}
	}
	
	/**
	 * Sorts candidate products for recommending. Uses the scores for the given Scoring Function ID. 
	 * @param prodDataList An ArrayList of ProdRecData objects.
	 * @param sfID The ID of the Scoring Function scores to use in sorting.
	 * @return void
	 * @roseuid 3A916AF6008C
	 */
	public void sortProducts(java.util.ArrayList prodDataList, final int sfID) 
	{
		final String mName = "sortProducts";
		ProdRecData.useSF(sfID);
		Object[] objAr = prodDataList.toArray();
		Arrays.sort(objAr);
		prodDataList.clear();
		for (int idx = 0; idx < objAr.length; idx ++) {
			prodDataList.add(objAr[idx]);
		}
//		prodDataList = null;
//		prodDataList = new ArrayList(Arrays.asList(objAr));
	}
}
